<?php
class Deal
{
    public $conn;
    private $table_name = "deals";

    // Object Properties
    public $id;
    public $product_id;
    public $title;
    public $description;
    public $discount_percentage;
    public $original_price;
    public $discounted_price;
    public $is_flash_deal;
    public $flash_deal_hours;
    public $social_requirements;
    public $is_active;

    public function __construct($db)
    {
        $this->conn = $db;
    }

    // Generate a new UUID
    private function generateUuid()
    {
        return sprintf(
            '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff)
        );
    }

    // Get a single deal by its ID
    public function getDealByIdForAdmin($id)
    {
        $query = "SELECT d.*, p.name as product_name, p.brand as product_brand
                  FROM " . $this->table_name . " d
                  LEFT JOIN products p ON d.product_id = p.id
                  WHERE d.id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        $deal = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($deal && $deal['social_requirements']) {
            $deal['social_requirements'] = json_decode($deal['social_requirements'], true);
        }
        
        return $deal;
    }

    // Create a new deal
    public function create()
    {
        $this->id = $this->generateUuid();
        
        $query = "INSERT INTO " . $this->table_name . "
                  SET id = :id, product_id = :product_id, title = :title, description = :description,
                      discount_percentage = :discount_percentage, original_price = :original_price,
                      discounted_price = :discounted_price, is_flash_deal = :is_flash_deal,
                      flash_deal_hours = :flash_deal_hours, social_requirements = :social_requirements";

        $stmt = $this->conn->prepare($query);

        // Sanitize data
        $this->product_id = htmlspecialchars(strip_tags($this->product_id));
        $this->title = htmlspecialchars(strip_tags($this->title));
        $this->description = htmlspecialchars(strip_tags($this->description));

        // Bind data
        $stmt->bindParam(':id', $this->id);
        $stmt->bindParam(':product_id', $this->product_id);
        $stmt->bindParam(':title', $this->title);
        $stmt->bindParam(':description', $this->description);
        $stmt->bindParam(':discount_percentage', $this->discount_percentage);
        $stmt->bindParam(':original_price', $this->original_price);
        $stmt->bindParam(':discounted_price', $this->discounted_price);
        $stmt->bindParam(':is_flash_deal', $this->is_flash_deal, PDO::PARAM_BOOL);
        $stmt->bindParam(':flash_deal_hours', $this->flash_deal_hours);
        $stmt->bindParam(':social_requirements', $this->social_requirements);

        if ($stmt->execute()) {
            return $this->id;
        }
        return false;
    }

    // Update an existing deal
    public function update()
    {
        $query = "UPDATE " . $this->table_name . "
                  SET title = :title, description = :description, discount_percentage = :discount_percentage,
                      original_price = :original_price, discounted_price = :discounted_price,
                      is_flash_deal = :is_flash_deal, flash_deal_hours = :flash_deal_hours,
                      is_active = :is_active" .
                 (isset($this->social_requirements) ? ", social_requirements = :social_requirements" : "") .
                 " WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);

        // Sanitize and bind data
        $stmt->bindParam(':id', $this->id);
        $stmt->bindParam(':title', $this->title);
        $stmt->bindParam(':description', $this->description);
        $stmt->bindParam(':discount_percentage', $this->discount_percentage);
        $stmt->bindParam(':original_price', $this->original_price);
        $stmt->bindParam(':discounted_price', $this->discounted_price);
        $stmt->bindParam(':is_flash_deal', $this->is_flash_deal, PDO::PARAM_BOOL);
        $stmt->bindParam(':flash_deal_hours', $this->flash_deal_hours);
        $stmt->bindParam(':is_active', $this->is_active, PDO::PARAM_BOOL);
        if (isset($this->social_requirements)) {
            $stmt->bindParam(':social_requirements', $this->social_requirements);
        }

        return $stmt->execute();
    }

    // Delete a deal
    public function delete()
    {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $this->id);
        return $stmt->execute();
    }
}
?>